package ga.view.config;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;

/**
 * Abstract class that handles setting using a {@link HashMap}.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public abstract class AbstractSettings {
  private final Map<String, Object> values = new HashMap<String, Object>();

  /**
   * Puts an object into the map.
   * 
   * @param id
   *          the id
   * @param o
   *          the Object
   * @return the Object
   * 
   * @see HashMap#put(Object, Object)
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public Object put(final String id, final Object o) {
    return values.put(id, o);
  }

  public int getInt(final String id, final int dfault) {
    try {
      return (Integer) values.get(id);
    } catch (final Exception e) {
      return dfault;
    }
  }

  /**
   * Clears the map of settings.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  protected void clear() {
    values.clear();
  }

  public Float getFloat(final String id, final float dfault) {
    try {
      return (Float) values.get(id);
    } catch (final Exception e) {
      return dfault;
    }
  }

  public String getString(final String id, final String dfault) {
    try {
      return (String) values.get(id);
    } catch (final Exception e) {
      return dfault;
    }
  }

  public Boolean getBoolean(final String id, final Boolean dfault) {
    try {
      return (Boolean) values.get(id);
    } catch (final Exception e) {
      return dfault;
    }
  }

  @SuppressWarnings("unchecked")
  public <T> T get(final String id, final T dfault) {
    try {
      return (T) values.get(id);
    } catch (final Exception e) {
      return dfault;
    }
  }

  public <T> T get(final String id) {
    return get(id, null);
  }

  /**
   * Creates data of ground datatype (boolean, string, int, long, float,
   * double).
   * 
   * @param type
   *          the type
   * @param value
   *          the value
   * @return the object
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  protected Object createSimple(final String type, final String value) {
    Object o = null;

    if ("string".equals(type)) {
      o = value;
    } else if ("int".equals(type)) {
      o = Integer.valueOf(value);
    } else if ("long".equals(type)) {
      o = Long.valueOf(value);
    } else if ("float".equals(type)) {
      o = Float.valueOf(value);
    } else if ("double".equals(type)) {
      o = Double.valueOf(value);
    } else if ("boolean".equals(type)) {
      o = Boolean.valueOf(value);
    }

    if (o == null) {
      throw new RuntimeException("Datatype unknown: " + type);
    }

    return o;
  }

  protected Object createObject(final Class<?> clss,
      final Class<?>[] parameterClasses, final Object[] parameters)
      throws ClassNotFoundException, NoSuchMethodException,
      InstantiationException, IllegalAccessException, InvocationTargetException {

    final Constructor<?> c = getConstructor(clss, parameterClasses);
    return c.newInstance(parameters);
  }

  protected Constructor<?> getConstructor(final Class<?> clss,
      final Class<?>[] parameterClasses) throws NoSuchMethodException {

    if (parameterClasses.length == 0) {
      return clss.getConstructor();
    }

    for (final Constructor<?> c : clss.getConstructors()) {
      final Class<?>[] parameterClasses2 = c.getParameterTypes();

      if (parameterClasses.length != parameterClasses2.length) {
        continue;
      }
      for (int i = 0; i < c.getParameterTypes().length; i++) {
        final Class<?> type = parameterClasses2[i];
        if (!parameterClasses2[i].isAssignableFrom(type)) {
          break;
        }
      }

      return c;
    }

    return null;
  }

  protected Object createObject(final String className,
      final Class<?>[] parameterClasses, final Object... parameters)
      throws ClassNotFoundException, NoSuchMethodException,
      InstantiationException, IllegalAccessException, InvocationTargetException {
    final Class<?> clss = Class.forName(className);

    final Object o = createObject(clss, parameterClasses, parameters);

    if (o == null) {
      throw new RuntimeException("Class unknown: " + className);
    }

    return o;
  }

  protected void putUnique(final String id, final Object o) {
    final Object old = values.put(id, o);
    if (old != null) {
      throw new RuntimeException("Value " + id + " was already set to " + old
          + " overriding with " + o);
    }
  }

  @Override
  public String toString() {
    return values.toString();
  }
}
